/*
 * Decompiled with CFR 0.152.
 */
package org.apache.jackrabbit.oak.segment.tool;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.io.PrintStream;
import java.io.PrintWriter;
import java.nio.file.CopyOption;
import java.nio.file.Files;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.ListIterator;
import java.util.Objects;
import java.util.Set;
import org.apache.jackrabbit.oak.commons.conditions.Validate;
import org.apache.jackrabbit.oak.segment.RecordId;
import org.apache.jackrabbit.oak.segment.RecordType;
import org.apache.jackrabbit.oak.segment.SegmentId;
import org.apache.jackrabbit.oak.segment.SegmentNodeState;
import org.apache.jackrabbit.oak.segment.SegmentNodeStore;
import org.apache.jackrabbit.oak.segment.SegmentNodeStoreBuilders;
import org.apache.jackrabbit.oak.segment.SegmentNotFoundException;
import org.apache.jackrabbit.oak.segment.file.ReadOnlyFileStore;
import org.apache.jackrabbit.oak.segment.file.tooling.ConsistencyChecker;
import org.apache.jackrabbit.oak.segment.tool.Utils;
import org.apache.jackrabbit.oak.spi.state.NodeState;

public class RecoverJournal {
    private final File path;
    private final PrintStream out;
    private final PrintStream err;
    private final Set<String> notFoundSegments = new HashSet<String>();

    public static Builder builder() {
        return new Builder();
    }

    private RecoverJournal(Builder builder) {
        this.path = builder.path;
        this.out = builder.out;
        this.err = builder.err;
    }

    public int run() {
        boolean rollback;
        List<Entry> entries;
        try (ReadOnlyFileStore store = Utils.openReadOnlyFileStore(this.path);){
            entries = this.recoverEntries(store);
        }
        catch (Exception e) {
            this.out.println("Unable to recover the journal entries, aborting");
            e.printStackTrace(this.err);
            return 1;
        }
        if (entries.size() == 0) {
            this.out.println("No valid journal entries found, aborting");
            return 1;
        }
        File journalBackup = this.journalBackupName();
        if (journalBackup == null) {
            this.err.println("Too many journal backups, please cleanup");
            return 1;
        }
        File journal = new File(this.path, "journal.log");
        try {
            Files.move(journal.toPath(), journalBackup.toPath(), new CopyOption[0]);
        }
        catch (IOException e) {
            this.err.println("Unable to backup old journal, aborting");
            e.printStackTrace(this.err);
            return 1;
        }
        this.out.printf("Old journal backed up at %s\n", journalBackup.getName());
        try (PrintWriter w = new PrintWriter(new BufferedWriter(new FileWriter(journal)));){
            for (Entry e : entries) {
                w.printf("%s root %d\n", e.recordId.toString10(), e.timestamp);
            }
            rollback = false;
        }
        catch (IOException e) {
            this.err.println("Unable to write the recovered journal, rolling back");
            e.printStackTrace(this.err);
            rollback = true;
        }
        if (rollback) {
            try {
                Files.deleteIfExists(journal.toPath());
            }
            catch (IOException e) {
                this.err.println("Unable to delete the recovered journal, aborting");
                e.printStackTrace(this.err);
                return 1;
            }
            try {
                Files.move(journalBackup.toPath(), journal.toPath(), new CopyOption[0]);
            }
            catch (IOException e) {
                this.err.println("Unable to roll back the old journal, aborting");
                e.printStackTrace(this.err);
                return 1;
            }
            this.out.println("Old journal rolled back");
            return 1;
        }
        this.out.println("Journal recovered");
        return 0;
    }

    private File journalBackupName() {
        for (int attempt = 0; attempt < 1000; ++attempt) {
            File backup = new File(this.path, String.format("journal.log.bak.%03d", attempt));
            if (backup.exists()) continue;
            return backup;
        }
        return null;
    }

    private List<Entry> recoverEntries(ReadOnlyFileStore fileStore) {
        ArrayList<Entry> entries = new ArrayList<Entry>();
        for (SegmentId segmentId : fileStore.getSegmentIds()) {
            try {
                this.recoverEntries(fileStore, segmentId, entries);
            }
            catch (SegmentNotFoundException e) {
                this.handle(e);
            }
        }
        entries.sort((left, right) -> {
            SegmentId rightSegmentId;
            int timestampComparison = Long.compare(left.timestamp, right.timestamp);
            if (timestampComparison != 0) {
                return timestampComparison;
            }
            SegmentId leftSegmentId = left.recordId.getSegmentId();
            int segmentIdComparison = leftSegmentId.compareTo(rightSegmentId = right.recordId.getSegmentId());
            if (segmentIdComparison != 0) {
                return segmentIdComparison;
            }
            int leftRecordNumber = left.recordId.getRecordNumber();
            int rightRecordNumber = right.recordId.getRecordNumber();
            return Integer.compare(leftRecordNumber, rightRecordNumber);
        });
        SegmentNodeStore nodeStore = SegmentNodeStoreBuilders.builder(fileStore).build();
        ConsistencyChecker checker = new ConsistencyChecker();
        HashSet<String> corruptedPaths = new HashSet<String>();
        ListIterator i = entries.listIterator(entries.size());
        block3: while (i.hasPrevious()) {
            Entry entry = (Entry)i.previous();
            fileStore.setRevision(entry.recordId.toString());
            String badHeadPath = checker.checkTreeConsistency(nodeStore.getRoot(), corruptedPaths, true);
            if (badHeadPath != null) {
                this.out.printf("Skipping revision %s, corrupted path in head: %s\n", entry.recordId, badHeadPath);
                corruptedPaths.add(badHeadPath);
                i.remove();
                continue;
            }
            for (String checkpoint : nodeStore.checkpoints()) {
                NodeState root = nodeStore.retrieve(checkpoint);
                if (root == null) {
                    this.out.printf("Skipping revision %s, found unreachable checkpoint %s\n", entry.recordId, checkpoint);
                    i.remove();
                    continue block3;
                }
                String badCheckpointPath = checker.checkTreeConsistency(root, corruptedPaths, true);
                if (badCheckpointPath == null) continue;
                this.out.printf("Skipping revision %s, corrupted path in checkpoint %s: %s\n", entry.recordId, checkpoint, badCheckpointPath);
                corruptedPaths.add(badCheckpointPath);
                i.remove();
                continue block3;
            }
        }
        return entries;
    }

    private void recoverEntries(ReadOnlyFileStore fileStore, SegmentId segmentId, List<Entry> entries) {
        if (segmentId.isBulkSegmentId()) {
            return;
        }
        Long timestamp = Utils.parseSegmentInfoTimestamp(segmentId);
        if (timestamp == null) {
            this.err.printf("No timestamp found in segment %s\n", segmentId);
            return;
        }
        segmentId.getSegment().forEachRecord((number, type, offset) -> {
            if (type != RecordType.NODE) {
                return;
            }
            try {
                this.recoverEntries(fileStore, timestamp, new RecordId(segmentId, number), entries);
            }
            catch (SegmentNotFoundException e) {
                this.handle(e);
            }
        });
    }

    private void recoverEntries(ReadOnlyFileStore fileStore, long timestamp, RecordId recordId, List<Entry> entries) {
        SegmentNodeState nodeState = fileStore.getReader().readNode(recordId);
        if (nodeState.hasChildNode("checkpoints") && nodeState.hasChildNode("root")) {
            entries.add(new Entry(timestamp, recordId));
        }
    }

    private void handle(SegmentNotFoundException e) {
        if (this.notFoundSegments.add(e.getSegmentId())) {
            e.printStackTrace(this.err);
        }
    }

    private static class Entry {
        long timestamp;
        RecordId recordId;

        Entry(long timestamp, RecordId recordId) {
            this.timestamp = timestamp;
            this.recordId = recordId;
        }
    }

    public static class Builder {
        private File path;
        private PrintStream out = System.out;
        private PrintStream err = System.err;

        private Builder() {
        }

        public Builder withPath(File path) {
            this.path = Objects.requireNonNull(path, "path");
            return this;
        }

        public Builder withOut(PrintStream out) {
            this.out = Objects.requireNonNull(out, "out");
            return this;
        }

        public Builder withErr(PrintStream err) {
            this.err = Objects.requireNonNull(err, "err");
            return this;
        }

        public RecoverJournal build() {
            Validate.checkState((this.path != null ? 1 : 0) != 0, (Object)"path not specified");
            return new RecoverJournal(this);
        }
    }
}

