/*
 * Decompiled with CFR 0.152.
 */
package org.silverpeas.kernel.logging;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.util.Collection;
import java.util.Map;
import java.util.Objects;
import java.util.Properties;
import java.util.Set;
import java.util.TreeSet;
import java.util.concurrent.ConcurrentHashMap;
import java.util.logging.Logger;
import java.util.stream.Collector;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.silverpeas.kernel.SilverpeasResourcesLocation;
import org.silverpeas.kernel.SilverpeasRuntimeException;
import org.silverpeas.kernel.annotation.Technical;
import org.silverpeas.kernel.logging.Level;

@Technical
public class LoggerConfigurationManager {
    private static final String LOGGER_CONF_FILE_SUFFIX = "Logging.properties";
    private static final String THIS_LOGGER_NAMESPACE = "silverpeas.core.logging";
    private static final String LOGGER_NAMESPACE = "namespace";
    private static final String LOGGER_LEVEL = "level";
    private static final int INITIAL_CAPACITY = 128;
    private static final Map<String, LoggerConfiguration> configs = new ConcurrentHashMap<String, LoggerConfiguration>(128);

    protected LoggerConfigurationManager() {
    }

    private Map<String, LoggerConfiguration> getLoggerConfigurations() {
        return configs;
    }

    protected void loadAllConfigurationFiles() {
        Logger.getLogger(THIS_LOGGER_NAMESPACE).log(java.util.logging.Level.INFO, "Silverpeas Logging Engine initialization...");
        Path configurationHome = SilverpeasResourcesLocation.getInstance().getLoggersRootPath();
        try (Stream<Path> loggers = Files.list(configurationHome);){
            configs.clear();
            configs.putAll(loggers.filter(p -> p.getFileName().toString().endsWith(LOGGER_CONF_FILE_SUFFIX)).map(this::loadLoggerConfiguration).collect(Collectors.toMap(LoggerConfiguration::getNamespace, l -> l)));
        }
        catch (IOException | SilverpeasRuntimeException e) {
            Logger.getLogger(THIS_LOGGER_NAMESPACE).log(java.util.logging.Level.WARNING, e.getMessage(), e);
        }
        if (configs.isEmpty()) {
            Logger.getLogger(THIS_LOGGER_NAMESPACE).log(java.util.logging.Level.WARNING, "No logging configuration files found for Silverpeas");
        }
    }

    public LoggerConfiguration getLoggerConfiguration(String namespace) {
        Map<String, LoggerConfiguration> loggerConfigurations = this.getLoggerConfigurations();
        return loggerConfigurations.computeIfAbsent(namespace, ns -> new LoggerConfiguration(null, (String)ns));
    }

    public void saveLoggerConfiguration(LoggerConfiguration configuration) {
        Map<String, LoggerConfiguration> loggerConfigurations = this.getLoggerConfigurations();
        if (loggerConfigurations.containsKey(configuration.getNamespace()) && configuration.hasConfigurationFile()) {
            LoggerConfiguration existing = loggerConfigurations.get(configuration.getNamespace());
            Properties properties = new Properties();
            try (InputStream input = Files.newInputStream(configuration.getConfigurationFile(), new OpenOption[0]);){
                properties.load(input);
            }
            catch (IOException e) {
                Logger.getLogger(THIS_LOGGER_NAMESPACE).log(java.util.logging.Level.WARNING, e.getMessage(), e);
            }
            if (configuration.getLevel() == null) {
                properties.remove(LOGGER_LEVEL);
                existing.setLevel(null);
            } else {
                properties.setProperty(LOGGER_LEVEL, configuration.getLevel().name());
                existing.setLevel(configuration.getLevel());
            }
            try (OutputStream output = Files.newOutputStream(configuration.getConfigurationFile(), new OpenOption[0]);){
                properties.store(output, null);
            }
            catch (IOException e) {
                Logger.getLogger(THIS_LOGGER_NAMESPACE).log(java.util.logging.Level.WARNING, e.getMessage(), e);
            }
        }
    }

    public Set<LoggerConfiguration> getAvailableLoggerConfigurations() {
        Collection<LoggerConfiguration> allConfigurations = this.getLoggerConfigurations().values();
        return allConfigurations.stream().filter(LoggerConfiguration::hasConfigurationFile).collect(Collector.of(TreeSet::new, TreeSet::add, (left, right) -> {
            left.addAll(right);
            return left;
        }, new Collector.Characteristics[0]));
    }

    private LoggerConfiguration loadLoggerConfiguration(Path loggerConfFile) {
        Properties loggerProperties = new Properties();
        try (InputStream input = Files.newInputStream(loggerConfFile, new OpenOption[0]);){
            loggerProperties.load(input);
        }
        catch (IOException e) {
            throw new SilverpeasRuntimeException(e);
        }
        String namespace = loggerProperties.getProperty(LOGGER_NAMESPACE);
        Level level = null;
        String strLevel = loggerProperties.getProperty(LOGGER_LEVEL);
        if (strLevel != null && !strLevel.trim().isEmpty()) {
            try {
                level = Level.valueOf(strLevel);
            }
            catch (Exception t) {
                Logger.getLogger(THIS_LOGGER_NAMESPACE).log(java.util.logging.Level.SEVERE, t.getMessage(), t);
            }
        }
        return new LoggerConfiguration(loggerConfFile, namespace).withLevel(level);
    }

    public static class LoggerConfiguration
    implements Comparable<LoggerConfiguration> {
        private final String namespace;
        private final Path file;
        private Level level;

        LoggerConfiguration(Path configFile, String namespace) {
            this.file = configFile;
            this.namespace = namespace;
        }

        public LoggerConfiguration withLevel(Level level) {
            this.setLevel(level);
            return this;
        }

        private Path getConfigurationFile() {
            return this.file;
        }

        protected boolean hasConfigurationFile() {
            return this.file != null && Files.exists(this.file, new LinkOption[0]) && Files.isRegularFile(this.file, new LinkOption[0]);
        }

        public String getNamespace() {
            return this.namespace;
        }

        public Level getLevel() {
            return this.level;
        }

        public void setLevel(Level level) {
            this.level = level;
        }

        @Override
        public int compareTo(LoggerConfiguration other) {
            return this.getNamespace().compareTo(other.getNamespace());
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (!(o instanceof LoggerConfiguration)) {
                return false;
            }
            LoggerConfiguration that = (LoggerConfiguration)o;
            return this.compareTo(that) == 0;
        }

        public int hashCode() {
            return Objects.hash(this.getNamespace());
        }
    }
}

