/*
 * Decompiled with CFR 0.152.
 */
package org.silverpeas.core.calendar;

import java.io.IOException;
import java.io.InputStream;
import java.lang.annotation.Annotation;
import java.text.MessageFormat;
import java.text.ParseException;
import java.time.Duration;
import java.time.Instant;
import java.time.OffsetDateTime;
import java.time.ZoneId;
import java.util.List;
import java.util.MissingResourceException;
import java.util.Optional;
import java.util.stream.Stream;
import javax.enterprise.event.Event;
import javax.enterprise.inject.Instance;
import javax.inject.Inject;
import javax.inject.Singleton;
import org.silverpeas.core.SilverpeasExceptionMessages;
import org.silverpeas.core.admin.user.model.User;
import org.silverpeas.core.annotation.Service;
import org.silverpeas.core.calendar.Calendar;
import org.silverpeas.core.calendar.ICalendarEventImportProcessor;
import org.silverpeas.core.calendar.ICalendarImportResult;
import org.silverpeas.core.importexport.ImportDescriptor;
import org.silverpeas.core.importexport.ImportException;
import org.silverpeas.core.initialization.Initialization;
import org.silverpeas.core.persistence.Transaction;
import org.silverpeas.core.persistence.datasource.OperationContext;
import org.silverpeas.core.scheduler.Job;
import org.silverpeas.core.scheduler.JobExecutionContext;
import org.silverpeas.core.scheduler.Scheduler;
import org.silverpeas.core.scheduler.SchedulerException;
import org.silverpeas.core.scheduler.trigger.JobTrigger;
import org.silverpeas.core.thread.ManagedThreadPool;
import org.silverpeas.core.thread.ManagedThreadPoolException;
import org.silverpeas.core.util.ServiceProvider;
import org.silverpeas.kernel.SilverpeasException;
import org.silverpeas.kernel.bundle.ResourceLocator;
import org.silverpeas.kernel.bundle.SettingBundle;
import org.silverpeas.kernel.logging.SilverLogger;

@Service
@Singleton
public class ICalendarEventSynchronization
implements Initialization {
    public static final String REPORT_NAMESPACE = "silverpeas.core.calendar.synchronization";
    private static final String CALENDAR_SETTINGS = "org.silverpeas.calendar.settings.calendar";
    private static final String SYNCHRONIZATION_ERROR_MSG = "Synchronize error on calendar {0} of instance {1} -> {2}";
    @Inject
    private ICalendarEventImportProcessor importer;
    @Inject
    private Event<CalendarBatchSynchronizationErrorEvent> notifier;
    @Inject
    private Instance<Scheduler> schedulerProvider;

    private ICalendarEventSynchronization() {
    }

    public static ICalendarEventSynchronization get() {
        return ServiceProvider.getService(ICalendarEventSynchronization.class, new Annotation[0]);
    }

    @Override
    public void init() {
        try {
            SettingBundle settings = ResourceLocator.getSettingBundle((String)CALENDAR_SETTINGS);
            String cron = settings.getString("calendar.synchronization.cron");
            Scheduler scheduler = (Scheduler)this.schedulerProvider.get();
            scheduler.unscheduleJob(this.getClass().getSimpleName());
            scheduler.scheduleJob(new Job(this.getClass().getSimpleName()){

                @Override
                public void execute(JobExecutionContext context) throws SilverpeasException {
                    ICalendarEventSynchronization.this.synchronizeAll();
                }
            }, JobTrigger.triggerAt(cron));
        }
        catch (ParseException | MissingResourceException | SchedulerException e) {
            SilverLogger.getLogger((Object)this).error("The synchronization scheduling failed to start", (Throwable)e);
        }
    }

    @Override
    public void release() throws SchedulerException {
        Scheduler scheduler = (Scheduler)this.schedulerProvider.get();
        scheduler.unscheduleJob(this.getClass().getSimpleName());
    }

    public ICalendarImportResult synchronize(Calendar calendar) throws ImportException {
        if (!calendar.isSynchronized() || !calendar.isPersisted()) {
            throw new IllegalArgumentException("The calendar " + calendar.getId() + " isn't a synchronized!");
        }
        try {
            InputStream source = (InputStream)calendar.getExternalCalendarUrl().getContent();
            return this.synchronizeCalendarFrom(calendar, source);
        }
        catch (IOException e) {
            String message = MessageFormat.format(SYNCHRONIZATION_ERROR_MSG, calendar.getId(), calendar.getComponentInstanceId(), "no data found from the synchronization link");
            throw new ImportException(new SilverpeasExceptionMessages.LightExceptionMessage(this, (Exception)e).singleLineWith(message));
        }
        catch (Exception e) {
            String message = MessageFormat.format(SYNCHRONIZATION_ERROR_MSG, calendar.getId(), calendar.getComponentInstanceId(), e.getMessage());
            throw new ImportException(message, e);
        }
    }

    private ICalendarImportResult synchronizeCalendarFrom(Calendar calendar, InputStream source) {
        return Transaction.performInOne(() -> {
            Instant synchronizationDateTime = Instant.now();
            calendar.setLastSynchronizationDate(synchronizationDateTime);
            Calendar syncCalendar = Calendar.getById(calendar.getId());
            syncCalendar.setLastSynchronizationDate(synchronizationDateTime);
            syncCalendar.save();
            ICalendarImportResult result = this.importer.importInto(syncCalendar, ImportDescriptor.withInputStream(source));
            this.removeDeletedEvents(syncCalendar, result);
            return result;
        });
    }

    public void synchronizeAll() throws ImportException {
        SettingBundle settings = ResourceLocator.getSettingBundle((String)CALENDAR_SETTINGS);
        int processors = settings.getInteger("calendar.synchronization.processors", 0);
        if (processors <= 0) {
            processors = Runtime.getRuntime().availableProcessors();
        }
        List<Calendar> calendars = Calendar.getSynchronizedCalendars();
        try {
            ManagedThreadPool.getPool().invokeAndAwaitTermination(this.synchronizationProcessorsOf(calendars), ManagedThreadPool.ExecutionConfig.maxThreadPoolSizeOf(processors));
        }
        catch (ManagedThreadPoolException e) {
            throw new ImportException("Fail to synchronize the synchronized calendars!", e);
        }
    }

    private void removeDeletedEvents(Calendar calendar, ICalendarImportResult result) {
        if (calendar.getLastSynchronizationDate().isPresent()) {
            Calendar.getEvents().filter(f -> f.onCalendar(calendar).onSynchronizationDateLimit(calendar.getLastSynchronizationDate().get())).stream().forEach(e -> {
                e.delete();
                result.incDeleted();
            });
        }
    }

    private Stream<? extends Runnable> synchronizationProcessorsOf(List<Calendar> calendars) {
        return calendars.stream().map(c -> () -> {
            try {
                User currentUser = User.getCurrentRequester();
                OperationContext.fromUser(currentUser != null ? currentUser : c.getCreator());
                ICalendarImportResult result = c.synchronize();
                String report = this.generateReport((Calendar)c, result);
                SilverLogger.getLogger((String)REPORT_NAMESPACE).info(report, new Object[0]);
            }
            catch (ImportException e) {
                if (e.getCause() != null) {
                    SilverLogger.getLogger((String)REPORT_NAMESPACE).error((Throwable)((Object)e));
                } else {
                    SilverLogger.getLogger((String)REPORT_NAMESPACE).error(e.getMessage(), new Object[0]);
                }
                this.notifier.fire((Object)new CalendarBatchSynchronizationErrorEvent((Calendar)c));
            }
        });
    }

    private String generateReport(Calendar calendar, ICalendarImportResult result) {
        String duration;
        String synchroDate;
        Optional<Instant> lastSynchroDate = calendar.getLastSynchronizationDate();
        if (lastSynchroDate.isPresent()) {
            Instant dateTime = lastSynchroDate.get();
            synchroDate = OffsetDateTime.ofInstant(dateTime, ZoneId.systemDefault()).toString();
            duration = String.valueOf(Duration.between(dateTime, Instant.now()).getSeconds());
        } else {
            duration = "N/A";
            synchroDate = "N/A";
        }
        return "Report of the synchronization of calendar " + calendar.getId() + " ('" + calendar.getTitle() + "')\nauthor: " + calendar.getCreator().getDisplayedName() + " (id '" + calendar.getCreator().getDisplayedName() + "')\nSynchronization date: " + synchroDate + "\nSynchronization duration: " + duration + " seconds\nNumber of events added: " + result.added() + "\nNumber of events updated: " + result.updated() + "\nNumber of events deleted: " + result.deleted() + "\n";
    }

    public static class CalendarBatchSynchronizationErrorEvent {
        private final Calendar calendar;

        private CalendarBatchSynchronizationErrorEvent(Calendar calendar) {
            this.calendar = calendar;
        }

        public Calendar getCalendar() {
            return this.calendar;
        }
    }
}

