/*
 * Decompiled with CFR 0.152.
 */
package org.silverpeas.core.date;

import java.io.Serializable;
import java.time.Instant;
import java.time.LocalDate;
import java.time.OffsetDateTime;
import java.time.ZoneOffset;
import java.time.ZonedDateTime;
import java.time.chrono.ChronoLocalDate;
import java.time.temporal.ChronoUnit;
import java.time.temporal.Temporal;
import java.util.Objects;
import javax.persistence.Column;
import javax.persistence.Embeddable;
import org.apache.commons.lang3.builder.HashCodeBuilder;
import org.silverpeas.core.annotation.constraint.DateRange;
import org.silverpeas.core.date.TemporalConverter;
import org.silverpeas.core.date.TimeUnit;

@Embeddable
@DateRange(start="startDate", end="endDate")
public class Period
implements Serializable {
    private static final long serialVersionUID = -4679172808271849961L;
    private static final Period INDEFINITE = Period.betweenInDays(Instant.MIN, Instant.MAX);
    @Column(name="startDate", nullable=false)
    private Instant startDateTime;
    @Column(name="endDate", nullable=false)
    private Instant endDateTime;
    @Column(name="inDays", nullable=false)
    private boolean inDays = false;

    public static Period indefinite() {
        return INDEFINITE;
    }

    public static Period between(Temporal start, Temporal end) {
        Objects.requireNonNull(start);
        Objects.requireNonNull(end);
        if (!start.getClass().equals(end.getClass())) {
            throw new IllegalArgumentException("Temporal parameters must be of same type. Actually period start is " + start.getClass().getSimpleName() + " and the period end is " + end.getClass().getSimpleName());
        }
        if (start instanceof LocalDate) {
            return Period.between(LocalDate.from(start), LocalDate.from(end));
        }
        return Period.between(TemporalConverter.asInstant(start), TemporalConverter.asInstant(end));
    }

    public static Period between(LocalDate startDay, LocalDate endDay) {
        Objects.requireNonNull(startDay);
        Objects.requireNonNull(endDay);
        LocalDate upperBound = startDay.equals(endDay) ? endDay.plusDays(1L) : endDay;
        return Period.betweenInDays(TemporalConverter.asInstant(startDay), TemporalConverter.asInstant(upperBound));
    }

    public static Period between(OffsetDateTime startDateTime, OffsetDateTime endDateTime) {
        return Period.between(TemporalConverter.asInstant(startDateTime), TemporalConverter.asInstant(endDateTime));
    }

    public static Period between(ZonedDateTime startDateTime, ZonedDateTime endDateTime) {
        return Period.between(TemporalConverter.asInstant(startDateTime), TemporalConverter.asInstant(endDateTime));
    }

    public static Period between(Instant startInstant, Instant endInstant) {
        Period.checkPeriod(startInstant, endInstant);
        Period period = new Period();
        period.startDateTime = Period.minOrInstant(startInstant);
        period.endDateTime = Period.maxOrInstant(endInstant);
        period.inDays = period.startsAtMinDate() && period.endsAtMaxDate();
        return period;
    }

    public static Period betweenInDays(Instant startInstant, Instant endInstant) {
        Period period = Period.between(startInstant, endInstant);
        period.inDays = true;
        return period;
    }

    public static Period betweenNullable(Temporal start, Temporal end) {
        if (start != null && end != null) {
            return Period.between(start, end);
        }
        if (start instanceof LocalDate || end instanceof LocalDate || start == null && end == null) {
            return Period.betweenInDays(Period.minOrInstant(start), Period.maxOrInstant(end));
        }
        return Period.between(Period.minOrInstant(start), Period.maxOrInstant(end));
    }

    public static Period betweenNullable(LocalDate startDay, LocalDate endDay) {
        LocalDate upperBound = startDay != null && startDay.equals(endDay) ? endDay.plusDays(1L) : endDay;
        return Period.betweenInDays(Period.minOrInstant(startDay), Period.maxOrInstant(upperBound));
    }

    public static Period betweenNullable(OffsetDateTime startDateTime, OffsetDateTime endDateTime) {
        return Period.between(Period.minOrInstant(startDateTime), Period.maxOrInstant(endDateTime));
    }

    public static Period betweenNullable(ZonedDateTime startDateTime, ZonedDateTime endDateTime) {
        return Period.between(Period.minOrInstant(startDateTime), Period.maxOrInstant(endDateTime));
    }

    public static Period betweenNullable(Instant startInstant, Instant endInstant) {
        Instant start = Period.minOrInstant(startInstant);
        Instant end = Period.maxOrInstant(endInstant);
        return Period.between(start, end);
    }

    public static PeriodBuilder from(Temporal temporal) {
        return new PeriodBuilder(TemporalConverter.asInstant(temporal));
    }

    public Temporal getStartDate() {
        Comparable<ChronoLocalDate> startDate = this.startsAtMinDate() ? (this.isInDays() ? LocalDate.MIN : OffsetDateTime.MIN) : (this.isInDays() ? LocalDate.ofInstant(this.startDateTime, ZoneOffset.UTC) : OffsetDateTime.ofInstant(this.startDateTime, ZoneOffset.UTC));
        return startDate;
    }

    public Temporal getEndDate() {
        Comparable<ChronoLocalDate> endDate = this.endsAtMaxDate() ? (this.isInDays() ? LocalDate.MAX : OffsetDateTime.MAX) : (this.isInDays() ? LocalDate.ofInstant(this.endDateTime, ZoneOffset.UTC) : OffsetDateTime.ofInstant(this.endDateTime, ZoneOffset.UTC));
        return endDate;
    }

    public boolean isInDays() {
        return this.inDays;
    }

    public boolean spanOverSeveralDays() {
        LocalDate endDate = this.isInDays() ? LocalDate.ofInstant(this.endDateTime, ZoneOffset.UTC).minusDays(1L) : LocalDate.ofInstant(this.endDateTime, ZoneOffset.UTC);
        LocalDate startDate = LocalDate.ofInstant(this.startDateTime, ZoneOffset.UTC);
        return endDate.isAfter(startDate);
    }

    public boolean isIndefinite() {
        return this.startsAtMinDate() && this.endsAtMaxDate();
    }

    public boolean startsAtMinDate() {
        return this.startDateTime.equals(Instant.MIN);
    }

    public boolean endsAtMaxDate() {
        return this.endDateTime.equals(Instant.MAX);
    }

    public boolean includes(Temporal dateTime) {
        Instant dt = TemporalConverter.asInstant(dateTime);
        return dt.compareTo(this.startDateTime) >= 0 && dt.compareTo(this.endDateTime) < 0;
    }

    public boolean endsBefore(Temporal dateTime) {
        Instant dt = TemporalConverter.asInstant(dateTime);
        return dt.compareTo(this.endDateTime) >= 0;
    }

    public boolean endsAfter(Temporal dateTime) {
        Instant dt = TemporalConverter.asInstant(dateTime);
        return dt.compareTo(this.endDateTime) < 0;
    }

    public boolean startsAfter(Temporal dateTime) {
        Instant dt = TemporalConverter.asInstant(dateTime);
        return dt.compareTo(this.startDateTime) < 0;
    }

    private static void checkPeriod(Instant startDateTime, Instant endDateTime) {
        Objects.requireNonNull(startDateTime);
        Objects.requireNonNull(endDateTime);
        if (startDateTime.isAfter(endDateTime) || startDateTime.equals(endDateTime)) {
            throw new IllegalArgumentException("The end datetime must be after the start datetime");
        }
    }

    private static Instant minOrInstant(Temporal temporal) {
        return temporal == null ? Instant.MIN : TemporalConverter.asInstant(temporal);
    }

    private static Instant maxOrInstant(Temporal temporal) {
        return temporal == null ? Instant.MAX : TemporalConverter.asInstant(temporal);
    }

    public Period copy() {
        Period period = new Period();
        period.startDateTime = this.startDateTime;
        period.endDateTime = this.endDateTime;
        period.inDays = this.inDays;
        return period;
    }

    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (!(o instanceof Period)) {
            return false;
        }
        Period period = (Period)o;
        return this.inDays == period.inDays && this.startDateTime.equals(period.startDateTime) && this.endDateTime.equals(period.endDateTime);
    }

    public int hashCode() {
        return new HashCodeBuilder().append((Object)this.startDateTime).append((Object)this.endDateTime).append(this.inDays).toHashCode();
    }

    public static class PeriodBuilder {
        private final Instant startDate;

        private PeriodBuilder(Instant startDate) {
            this.startDate = startDate;
        }

        public Period of(long amount, TimeUnit unit) {
            ChronoUnit chrono = unit.toChronoUnit();
            Instant endDate = this.startDate.plus(amount, chrono);
            Period period = chrono.isDateBased() ? Period.betweenInDays(this.startDate, endDate) : Period.between(this.startDate, endDate);
            return period;
        }

        public Period of(long amount, ChronoUnit unit) {
            Instant endDate = this.startDate.plus(amount, unit);
            Period period = unit.isDateBased() ? Period.betweenInDays(this.startDate, endDate) : Period.between(this.startDate, endDate);
            return period;
        }

        public Period ofHours(long amount) {
            return this.of(amount, ChronoUnit.HOURS);
        }

        public Period ofDays(long amount) {
            return this.of(amount, ChronoUnit.DAYS);
        }

        public Period ofWeeks(long amount) {
            return this.of(amount, ChronoUnit.WEEKS);
        }

        public Period ofMonths(long amount) {
            return this.of(amount, ChronoUnit.MONTHS);
        }

        public Period ofYears(long amount) {
            return this.of(amount, ChronoUnit.YEARS);
        }
    }
}

