/*
 * Decompiled with CFR 0.152.
 */
package org.silverpeas.core.util.file;

import java.io.File;
import java.io.FileFilter;
import java.io.IOException;
import java.nio.charset.Charset;
import java.nio.file.CopyOption;
import java.nio.file.FileVisitOption;
import java.nio.file.Files;
import java.nio.file.InvalidPathException;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardCopyOption;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.MissingResourceException;
import java.util.Objects;
import java.util.StringTokenizer;
import java.util.stream.Stream;
import javax.activation.MimetypesFileTypeMap;
import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.FilenameUtils;
import org.apache.commons.io.filefilter.FileFilterUtils;
import org.apache.commons.io.filefilter.IOFileFilter;
import org.silverpeas.core.SilverpeasExceptionMessages;
import org.silverpeas.core.cache.service.CacheAccessorProvider;
import org.silverpeas.core.exception.RelativeFileAccessException;
import org.silverpeas.core.io.media.MetadataExtractor;
import org.silverpeas.core.mail.extractor.Mail;
import org.silverpeas.core.util.ImageUtil;
import org.silverpeas.core.util.MimeTypes;
import org.silverpeas.core.util.OsEnum;
import org.silverpeas.core.util.file.FileRepositoryManager;
import org.silverpeas.kernel.annotation.NonNull;
import org.silverpeas.kernel.bundle.ResourceLocator;
import org.silverpeas.kernel.bundle.SettingBundle;
import org.silverpeas.kernel.logging.SilverLogger;
import org.silverpeas.kernel.util.StringUtil;

public class FileUtil {
    private static final SettingBundle MIME_TYPES_EXTENSIONS = ResourceLocator.getSettingBundle((String)"org.silverpeas.util.attachment.mime_types");
    public static final String CONTEXT_TOKEN = ",";
    public static final String BASE_CONTEXT = "Attachment";
    private static final MimetypesFileTypeMap MIME_TYPES = new MimetypesFileTypeMap();
    private static final String MIME_TYPE_CACHE_KEY_PREFIX = "FileUtil.getMimeType$$";

    private FileUtil() {
    }

    @NonNull
    public static String getMimeType(String fileName) {
        String cacheKey = MIME_TYPE_CACHE_KEY_PREFIX + fileName;
        String cachedMimeType = (String)CacheAccessorProvider.getThreadCacheAccessor().getCache().get((Object)cacheKey, String.class);
        if (StringUtil.isDefined((String)cachedMimeType)) {
            return cachedMimeType;
        }
        String mimeType = FileUtil.computeMimeType(fileName);
        CacheAccessorProvider.getThreadCacheAccessor().getCache().put((Object)cacheKey, (Object)mimeType);
        return mimeType;
    }

    @NonNull
    private static String computeMimeType(String fileName) {
        String mimeType = null;
        File file = new File(fileName);
        if (Files.exists(file.toPath(), new LinkOption[0])) {
            mimeType = FileUtil.getMimeTypeByMetadata(file);
        }
        String fileExtension = FileRepositoryManager.getFileExtension(fileName).toLowerCase();
        if (!StringUtil.isDefined((String)mimeType) && MIME_TYPES_EXTENSIONS != null && !fileExtension.isEmpty()) {
            mimeType = FileUtil.getMimeTypeByFileExtension(fileExtension, mimeType);
        }
        if (!StringUtil.isDefined((String)mimeType)) {
            mimeType = MIME_TYPES.getContentType(fileName);
        }
        if ((mimeType = FileUtil.getPeculiarChildMimeType(mimeType, fileExtension)) == null) {
            mimeType = "application/octet-stream";
        }
        return mimeType;
    }

    @Nullable
    private static String getPeculiarChildMimeType(String parentMimeType, String fileExtension) {
        String mimeType = parentMimeType;
        if ("application/xhtml+xml".equalsIgnoreCase(parentMimeType) || "text/html".equalsIgnoreCase(parentMimeType)) {
            if (fileExtension.contains("jsp")) {
                mimeType = "text/x-jsp";
            } else if (fileExtension.contains("php")) {
                mimeType = "text/x-php";
            }
        } else if ("application/x-zip-compressed".equalsIgnoreCase(parentMimeType) || "application/zip".equalsIgnoreCase(parentMimeType)) {
            if ("jar".equalsIgnoreCase(fileExtension) || "war".equalsIgnoreCase(fileExtension) || "ear".equalsIgnoreCase(fileExtension)) {
                mimeType = "application/java-archive";
            } else if ("3D".equalsIgnoreCase(fileExtension)) {
                mimeType = "application/xview3d-3d";
            }
        }
        return mimeType;
    }

    private static String getMimeTypeByFileExtension(String fileExtension, String defaultMimeType) {
        String mimeType = defaultMimeType;
        try {
            mimeType = MIME_TYPES_EXTENSIONS.getString(fileExtension);
        }
        catch (MissingResourceException e) {
            SilverLogger.getLogger(FileUtil.class).warn("Unknown mime-type: {0}", new Object[]{e.getMessage(), e});
        }
        return mimeType;
    }

    private static String getMimeTypeByMetadata(File file) {
        String mimeType = null;
        try {
            mimeType = MetadataExtractor.get().detectMimeType(file);
        }
        catch (Exception ex) {
            SilverLogger.getLogger(FileUtil.class).warn("File exists ({0}), but mime-type has been detected: {1}", new Object[]{file.getName(), ex.getMessage(), ex});
        }
        return mimeType;
    }

    public static String[] getAttachmentContext(String context) {
        if (!StringUtil.isDefined((String)context)) {
            return new String[]{BASE_CONTEXT};
        }
        StringTokenizer strToken = new StringTokenizer(context, CONTEXT_TOKEN);
        ArrayList<String> folders = new ArrayList<String>(10);
        folders.add(BASE_CONTEXT);
        while (strToken.hasMoreElements()) {
            folders.add(strToken.nextToken().trim());
        }
        return folders.toArray(new String[0]);
    }

    public static String readFileToString(File file) throws IOException {
        return Files.readString(file.toPath(), Charset.defaultCharset());
    }

    public static boolean isWindows() {
        return OsEnum.getOS().isWindows();
    }

    public static boolean isSpinfireDocument(String filename) {
        return "application/xview3d-3d".equals(FileUtil.getMimeType(filename));
    }

    public static boolean isArchive(String filename) {
        return MimeTypes.ARCHIVE_MIME_TYPES.contains(FileUtil.getMimeType(filename));
    }

    public static boolean isImage(String filename) {
        String mimeType = FileUtil.getMimeType(filename);
        if ("application/octet-stream".equals(mimeType)) {
            return FilenameUtils.isExtension((String)filename.toLowerCase(), (String[])ImageUtil.IMAGE_EXTENTIONS);
        }
        return mimeType.startsWith("image");
    }

    public static boolean isMail(String filename) {
        return FilenameUtils.isExtension((String)filename, (String[])Mail.MAIL_EXTENTIONS);
    }

    public static boolean isPdf(String filename) {
        String mimeType = FileUtil.getMimeType(filename);
        return "application/pdf".equals(mimeType);
    }

    public static boolean isOpenOfficeCompatible(String filename) {
        String mimeType = FileUtil.getMimeType(filename);
        return MimeTypes.OPEN_OFFICE_MIME_TYPES.contains(mimeType) || FileUtil.isMsOfficeExtension(mimeType);
    }

    private static boolean isMsOfficeExtension(String mimeType) {
        return mimeType.startsWith("application/vnd.ms-word") || mimeType.startsWith("application/vnd.ms-excel") || mimeType.startsWith("application/vnd.ms-powerpoint") || mimeType.startsWith("application/vnd.ms-project");
    }

    public static void assertPathNotRelative(String path) throws RelativeFileAccessException {
        String unixPath = FilenameUtils.separatorsToUnix((String)path);
        if (unixPath != null && (unixPath.contains("../") || unixPath.contains("/.."))) {
            throw new RelativeFileAccessException(SilverpeasExceptionMessages.failureOnGetting((String)"path with relative parts", (Object)path), new String[0]);
        }
    }

    public static void forceDeletion(File fileToDelete) throws IOException {
        boolean result;
        if (fileToDelete.exists() && !fileToDelete.canWrite() && !(result = fileToDelete.setWritable(true))) {
            SilverLogger.getLogger(FileUtil.class).warn("Cannot set file ${0} writable", new Object[]{fileToDelete.getName()});
        }
        try (Stream<Path> paths = Files.walk(fileToDelete.toPath(), new FileVisitOption[0]);){
            paths.sorted(Comparator.reverseOrder()).map(Path::toFile).forEach(File::delete);
        }
    }

    public static void moveFile(File source, File destination) throws IOException {
        if (destination.exists()) {
            FileUtil.forceDeletion(destination);
        }
        Files.move(source.toPath(), destination.toPath(), new CopyOption[0]);
    }

    public static void copyFile(File source, File destination) throws IOException {
        boolean result;
        if (destination.exists() && !destination.canWrite() && !(result = destination.setWritable(true))) {
            SilverLogger.getLogger(FileUtil.class).warn("Cannot set file ${0} writable", new Object[]{destination.getName()});
        }
        Files.copy(source.toPath(), destination.toPath(), StandardCopyOption.REPLACE_EXISTING);
    }

    public static String getFilename(String fileName) {
        if (!StringUtil.isDefined((String)fileName)) {
            return "";
        }
        return FilenameUtils.getName((String)fileName);
    }

    public static String convertPathToServerOS(String undeterminedOsPath) {
        if (!StringUtil.isDefined((String)undeterminedOsPath)) {
            return "";
        }
        String localPath = undeterminedOsPath;
        localPath = localPath.replace('\\', File.separatorChar).replace('/', File.separatorChar);
        return localPath;
    }

    public static boolean deleteEmptyDir(@Nonnull File directory) {
        if (directory.exists() && directory.isDirectory() && directory.list() != null && Objects.requireNonNull(directory.list()).length == 0) {
            try {
                Files.delete(directory.toPath());
                return true;
            }
            catch (IOException e) {
                SilverLogger.getLogger(FileUtil.class).warn((Throwable)e);
                return false;
            }
        }
        return false;
    }

    public static File[] moveAllFilesAtRootFolder(File rootFolder) throws IOException {
        return FileUtil.moveAllFilesAtRootFolder(rootFolder, true);
    }

    public static File[] moveAllFilesAtRootFolder(File rootFolder, boolean deleteFolders) throws IOException {
        File[] foldersAtRoot;
        File[] fileArray = foldersAtRoot = rootFolder != null ? rootFolder.listFiles((FileFilter)FileFilterUtils.directoryFileFilter()) : null;
        if (foldersAtRoot != null) {
            for (File folderAtRoot : foldersAtRoot) {
                for (File file : FileUtils.listFiles((File)folderAtRoot, (IOFileFilter)FileFilterUtils.fileFileFilter(), (IOFileFilter)FileFilterUtils.trueFileFilter())) {
                    File newFilePath = new File(rootFolder, file.getName());
                    if (newFilePath.exists()) continue;
                    FileUtils.moveFile((File)file, (File)newFilePath);
                }
                if (!deleteFolders) continue;
                FileUtils.deleteQuietly((File)folderAtRoot);
            }
        }
        return foldersAtRoot != null ? foldersAtRoot : new File[]{};
    }

    public static void validateFilename(String fileName, String intendedDir) throws IOException {
        File iD;
        String canonicalID;
        File f = new File(fileName);
        String canonicalPath = f.getCanonicalPath();
        if (!canonicalPath.startsWith(canonicalID = (iD = new File(intendedDir)).getCanonicalPath())) {
            throw new IllegalStateException("File is outside extraction target directory (security)");
        }
    }

    public static String checkTaintedData(String value) {
        if (StringUtil.isDefined((String)value) && value.contains("..")) {
            throw new IllegalArgumentException(String.format("Value '%s' is forbidden", value));
        }
        return value;
    }

    public static void checkFileName(String fileName) {
        if (fileName == null || fileName.isEmpty() || fileName.length() > 255) {
            throw new InvalidPathException(fileName == null ? "" : fileName, "The file name is invalid: either it is empty or exceeds the size limit");
        }
        boolean isWindows = System.getProperty("os.name").toLowerCase().contains("win");
        if (isWindows && fileName.contains("\\") || fileName.contains("/")) {
            throw new IllegalArgumentException(String.format("File name '%s' isn't a filename but a path", fileName));
        }
        FileUtil.checkTaintedData(fileName);
        Paths.get(fileName, new String[0]);
    }
}

