/*
 * Decompiled with CFR 0.152.
 */
package org.apache.jackrabbit.oak.jcr.session;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.atomic.AtomicLong;
import java.util.regex.Pattern;
import java.util.regex.PatternSyntaxException;
import org.apache.jackrabbit.guava.common.base.Strings;
import org.apache.jackrabbit.oak.commons.json.JsonObject;
import org.apache.jackrabbit.oak.commons.json.JsopBuilder;
import org.apache.jackrabbit.oak.commons.json.JsopTokenizer;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SessionSaveDelayerConfig {
    private static final Logger LOG = LoggerFactory.getLogger(SessionSaveDelayerConfig.class);
    private final List<DelayEntry> entries;

    public SessionSaveDelayerConfig(@NotNull List<DelayEntry> entries) {
        this.entries = new ArrayList<DelayEntry>(entries);
    }

    @NotNull
    public static SessionSaveDelayerConfig fromJson(@NotNull String jsonConfig) throws IllegalArgumentException {
        if (Strings.isNullOrEmpty((String)jsonConfig)) {
            return new SessionSaveDelayerConfig(List.of());
        }
        try {
            JsopTokenizer tokenizer = new JsopTokenizer(jsonConfig);
            tokenizer.read(123);
            JsonObject root = JsonObject.create((JsopTokenizer)tokenizer);
            ArrayList<DelayEntry> entries = new ArrayList<DelayEntry>();
            String entriesJson = (String)root.getProperties().get("entries");
            if (entriesJson != null) {
                JsopTokenizer entryTokenizer = new JsopTokenizer(entriesJson);
                entryTokenizer.read(91);
                if (!entryTokenizer.matches(93)) {
                    do {
                        if (entryTokenizer.matches(123)) {
                            DelayEntry entry = SessionSaveDelayerConfig.parseDelayEntry(JsonObject.create((JsopTokenizer)entryTokenizer));
                            if (entry == null) continue;
                            entries.add(entry);
                            continue;
                        }
                        throw new IllegalArgumentException("Expected object in entries array");
                    } while (entryTokenizer.matches(44));
                    entryTokenizer.read(93);
                }
            }
            return new SessionSaveDelayerConfig(entries);
        }
        catch (Exception e) {
            throw new IllegalArgumentException("Failed to parse JSON configuration: " + e.getMessage(), e);
        }
    }

    public List<DelayEntry> getEntries() {
        return this.entries;
    }

    public long getDelayNanos(@NotNull String threadName, @Nullable String userData, @Nullable String stackTrace) {
        for (DelayEntry d : this.entries) {
            if (!d.matches(threadName, userData, stackTrace)) continue;
            return d.getDelayNanos();
        }
        return 0L;
    }

    @Nullable
    private static DelayEntry parseDelayEntry(JsonObject entryObj) {
        String delayMillis = (String)entryObj.getProperties().get("delayMillis");
        String threadNameRegex = (String)entryObj.getProperties().get("threadNameRegex");
        String userDataRegex = (String)entryObj.getProperties().get("userDataRegex");
        String stackTraceRegex = (String)entryObj.getProperties().get("stackTraceRegex");
        String maxSavesPerSecond = (String)entryObj.getProperties().get("maxSavesPerSecond");
        if (delayMillis == null || threadNameRegex == null) {
            LOG.warn("Skipping entry with missing required fields (delay or threadNameRegex)");
            return null;
        }
        try {
            double delay = Double.parseDouble(delayMillis);
            if (delay < 0.0) {
                LOG.warn("Skipping entry with negative delay");
                return null;
            }
            double maxSaves = 0.0;
            if (maxSavesPerSecond != null && (maxSaves = Double.parseDouble(maxSavesPerSecond)) < 0.0) {
                LOG.warn("Skipping entry with negative maxSavesPerSecond");
                return null;
            }
            Pattern threadPattern = Pattern.compile(JsopTokenizer.decodeQuoted((String)threadNameRegex));
            Pattern stackPattern = null;
            if (stackTraceRegex != null) {
                stackPattern = Pattern.compile(JsopTokenizer.decodeQuoted((String)stackTraceRegex));
            }
            Pattern userDataPattern = null;
            if (userDataRegex != null) {
                userDataPattern = Pattern.compile(JsopTokenizer.decodeQuoted((String)userDataRegex));
            }
            return new DelayEntry(delay, threadPattern, userDataPattern, stackPattern, maxSaves);
        }
        catch (NumberFormatException e) {
            LOG.warn("Skipping entry with invalid delay value or maxSavesPerSecond: {}", (Object)e.getMessage());
            return null;
        }
        catch (PatternSyntaxException e) {
            LOG.warn("Skipping entry with invalid regex pattern: {}", (Object)e.getMessage());
            return null;
        }
    }

    public String toString() {
        JsopBuilder json = new JsopBuilder();
        json.object().key("entries").array();
        for (DelayEntry entry : this.entries) {
            entry.toJson(json);
        }
        json.endArray().endObject();
        return JsopBuilder.prettyPrint((String)json.toString());
    }

    @Nullable
    public static String getCurrentStackTrace() {
        StackTraceElement[] stackTrace = Thread.currentThread().getStackTrace();
        if (stackTrace == null || stackTrace.length == 0) {
            return null;
        }
        StringBuilder sb = new StringBuilder();
        for (int i = 0; i < stackTrace.length; ++i) {
            if (i > 0) {
                sb.append("\n\tat ");
            } else {
                sb.append("at ");
            }
            sb.append(stackTrace[i]);
        }
        return sb.toString();
    }

    public static class DelayEntry {
        private final long baseDelayNanos;
        private final Pattern threadNamePattern;
        private final Pattern stackTracePattern;
        private final Pattern userDataPattern;
        private final double maxSavesPerSecond;
        private final AtomicLong lastMatch = new AtomicLong(0L);

        public DelayEntry(double delayMillis, @NotNull Pattern threadNamePattern, @Nullable Pattern userDataPattern, @Nullable Pattern stackTracePattern, double maxSavesPerSecond) {
            this.baseDelayNanos = (long)(delayMillis * 1000000.0);
            this.threadNamePattern = threadNamePattern;
            this.userDataPattern = userDataPattern;
            this.stackTracePattern = stackTracePattern;
            this.maxSavesPerSecond = maxSavesPerSecond;
        }

        public long getDelayNanos() {
            long totalDelayNanos = this.baseDelayNanos;
            if (this.maxSavesPerSecond > 0.0) {
                long nextAllowedTime;
                long currentTime = System.currentTimeMillis();
                double intervalMs = 1000.0 / this.maxSavesPerSecond;
                long lastMatchTime = this.lastMatch.get();
                if (lastMatchTime > 0L && currentTime < (nextAllowedTime = lastMatchTime + (long)intervalMs)) {
                    long rateLimitDelayMs = nextAllowedTime - currentTime;
                    totalDelayNanos += rateLimitDelayMs * 1000000L;
                }
                this.lastMatch.set(currentTime);
            }
            return totalDelayNanos;
        }

        public long getBaseDelayNanos() {
            return this.baseDelayNanos;
        }

        public double getMaxSavesPerSecond() {
            return this.maxSavesPerSecond;
        }

        @NotNull
        public Pattern getThreadNamePattern() {
            return this.threadNamePattern;
        }

        @Nullable
        public Pattern getStackTracePattern() {
            return this.stackTracePattern;
        }

        @Nullable
        public Pattern getUserDataPattern() {
            return this.userDataPattern;
        }

        boolean matches(@NotNull String threadName, @Nullable String userData, @Nullable String stackTrace) {
            if (!this.threadNamePattern.matcher(threadName).matches()) {
                return false;
            }
            if (this.userDataPattern != null) {
                if (userData == null) {
                    return false;
                }
                if (!this.userDataPattern.matcher(userData).find()) {
                    return false;
                }
            }
            if (this.stackTracePattern != null) {
                if (stackTrace == null) {
                    stackTrace = SessionSaveDelayerConfig.getCurrentStackTrace();
                }
                return this.stackTracePattern.matcher(stackTrace).find();
            }
            return true;
        }

        public String toString() {
            return this.toJson(new JsopBuilder()).toString();
        }

        public JsopBuilder toJson(JsopBuilder json) {
            json.object();
            double delayMillis = (double)this.baseDelayNanos / 1000000.0;
            json.key("delayMillis").encodedValue(Double.toString(delayMillis));
            json.key("threadNameRegex").value(this.threadNamePattern.pattern());
            if (this.userDataPattern != null) {
                json.key("userDataRegex").value(this.userDataPattern.pattern());
            }
            if (this.stackTracePattern != null) {
                json.key("stackTraceRegex").value(this.stackTracePattern.pattern());
            }
            if (this.maxSavesPerSecond > 0.0) {
                json.key("maxSavesPerSecond").encodedValue(Double.toString(this.maxSavesPerSecond));
            }
            return json.endObject();
        }
    }
}

