/*
 * Decompiled with CFR 0.152.
 */
package org.apache.jackrabbit.oak.plugins.blob;

import java.io.Closeable;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import java.util.stream.Stream;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.FilenameUtils;
import org.apache.jackrabbit.guava.common.cache.AbstractCache;
import org.apache.jackrabbit.guava.common.cache.Cache;
import org.apache.jackrabbit.guava.common.cache.CacheLoader;
import org.apache.jackrabbit.guava.common.cache.RemovalCause;
import org.apache.jackrabbit.guava.common.cache.Weigher;
import org.apache.jackrabbit.oak.cache.CacheLIRS;
import org.apache.jackrabbit.oak.commons.FileIOUtils;
import org.apache.jackrabbit.oak.commons.StringUtils;
import org.apache.jackrabbit.oak.commons.concurrent.ExecutorCloser;
import org.apache.jackrabbit.oak.commons.io.FileTreeTraverser;
import org.apache.jackrabbit.oak.commons.time.Stopwatch;
import org.apache.jackrabbit.oak.plugins.blob.DataStoreCacheStatsMBean;
import org.apache.jackrabbit.oak.plugins.blob.DataStoreCacheUpgradeUtils;
import org.apache.jackrabbit.oak.plugins.blob.DataStoreCacheUtils;
import org.apache.jackrabbit.oak.plugins.blob.FileCacheStats;
import org.jetbrains.annotations.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FileCache
extends AbstractCache<String, File>
implements Closeable {
    private static final Logger LOG = LoggerFactory.getLogger(FileCache.class);
    private static final int SEGMENT_COUNT = Integer.getInteger("oak.blob.fileCache.segmentCount", 1);
    private static final int MAX_ENTRY_COUNT = Integer.getInteger("oak.blob.fileCache.maxEntryCount", 500000);
    protected static final String DOWNLOAD_DIR = "download";
    private static final long ONE_SECOND_IN_MILLIS = 1000L;
    private static final AtomicLong lastLogMessage = new AtomicLong();
    private File parent;
    private File cacheRoot;
    private CacheLIRS<String, String> cache;
    private FileCacheStats cacheStats;
    private ExecutorService executor;
    private CacheLoader<String, String> cacheLoader;
    private Weigher<String, String> weigher;
    private Weigher<String, String> memWeigher;
    private long maxEntryCount = MAX_ENTRY_COUNT;
    private final long maxBlocks;
    private long currentBlockLimit;
    private long highWaterMark;
    private long loggedWaterMark;

    private FileCache(long maxSize, File root, final CacheLoader<String, InputStream> loader, @Nullable ExecutorService executor) {
        this.parent = root;
        this.cacheRoot = new File(root, DOWNLOAD_DIR);
        this.currentBlockLimit = this.maxBlocks = (long)Math.round(maxSize / 4096L);
        this.weigher = (key, value) -> {
            long value2 = this.getFile((String)key).length();
            return Math.round(value2 / 4096L) + 0;
        };
        this.memWeigher = (key, value) -> StringUtils.estimateMemoryUsage((String)key) + 128;
        this.cacheLoader = new CacheLoader<String, String>(){

            public String load(String key) throws Exception {
                File cachedFile = FileCache.this.getFile(key);
                if (cachedFile.exists()) {
                    return key;
                }
                long startNanos = System.nanoTime();
                try (InputStream is = (InputStream)loader.load((Object)key);){
                    FileIOUtils.copyInputStreamToFile((InputStream)is, (File)cachedFile);
                }
                catch (Exception e) {
                    LOG.warn("Error reading object for id [{}] from backend", (Object)key, (Object)e);
                    throw e;
                }
                if (LOG.isDebugEnabled()) {
                    LOG.debug("Loaded file: {} in {}", (Object)key, (Object)((System.nanoTime() - startNanos) / 1000000L));
                }
                return key;
            }
        };
        this.cache = new CacheLIRS.Builder().maximumWeight(this.maxBlocks).recordStats().weigher(this.weigher).segmentCount(SEGMENT_COUNT).evictionCallback((key, value, cause) -> {
            try {
                if (value != null && this.getFile((String)key).exists() && cause != RemovalCause.REPLACED) {
                    long last = lastLogMessage.get();
                    DataStoreCacheUtils.recursiveDelete(this.getFile((String)key), this.cacheRoot);
                    long now = System.currentTimeMillis();
                    if (now - last >= 1000L && lastLogMessage.compareAndSet(last, now)) {
                        Object reason = cause.toString();
                        if ("SIZE".equals(reason) && this.currentBlockLimit != this.maxBlocks) {
                            reason = "ENTRY_COUNT > " + this.maxEntryCount;
                        }
                        LOG.info("File [{}] evicted with reason [{}]", (Object)this.getFile((String)key), reason);
                    }
                }
            }
            catch (IOException e) {
                LOG.info("Cached file deletion failed after eviction", (Throwable)e);
            }
        }).build();
        this.cacheStats = new FileCacheStats((Cache<?, ?>)this.cache, this.weigher, this.memWeigher, maxSize);
        this.executor = executor == null ? Executors.newSingleThreadExecutor() : executor;
        this.executor.submit(new CacheBuildJob());
    }

    public void setMaxEntryCount(long maxEntryCount) {
        this.maxEntryCount = maxEntryCount;
    }

    public long getEntryCount() {
        return this.cache.size();
    }

    private FileCache() {
        this.maxBlocks = 0L;
    }

    private File getFile(String key) {
        return DataStoreCacheUtils.getFile(key, this.cacheRoot);
    }

    public static FileCache build(long maxSize, File root, CacheLoader<String, InputStream> loader, @Nullable ExecutorService executor) {
        if (maxSize > 0L) {
            return new FileCache(maxSize, root, loader, executor);
        }
        return new FileCache(){
            private final Cache<?, ?> cache = new CacheLIRS(0);

            @Override
            public void put(String key, File file) {
            }

            @Override
            public boolean containsKey(String key) {
                return false;
            }

            @Override
            @Nullable
            public File getIfPresent(String key) {
                return null;
            }

            @Override
            public File get(String key) {
                return null;
            }

            @Override
            public void invalidate(Object key) {
            }

            @Override
            public DataStoreCacheStatsMBean getStats() {
                return new FileCacheStats(this.cache, (key, value) -> 1, (key, value) -> 1, 0L);
            }

            @Override
            public void close() {
            }
        };
    }

    public void put(String key, File file) {
        this.adjustSize();
        this.put(key, file, true);
    }

    private void put(String key, File file, boolean copy) {
        try {
            File cached = DataStoreCacheUtils.getFile(key, this.cacheRoot);
            if (!cached.exists()) {
                if (copy) {
                    FileUtils.copyFile((File)file, (File)cached);
                } else {
                    FileUtils.moveFile((File)file, (File)cached);
                }
            }
            this.cache.put((Object)key, (Object)key);
        }
        catch (IOException e) {
            LOG.error("Exception adding id [{}] with file [{}] to cache, root cause: {}", new Object[]{key, file, e.getMessage()});
            LOG.debug("Root cause", (Throwable)e);
        }
    }

    public boolean containsKey(String key) {
        return this.cache.containsKey((Object)key);
    }

    @Nullable
    public File getIfPresent(String key) {
        try {
            String value = (String)this.cache.getIfPresent((Object)key);
            return value == null ? null : this.getFile(key);
        }
        catch (Exception e) {
            LOG.error("Error in retrieving [{}] from cache", (Object)key, (Object)e);
            return null;
        }
    }

    @Nullable
    public File getIfPresent(Object key) {
        return this.getIfPresent((String)key);
    }

    public File get(String key) throws IOException {
        this.adjustSize();
        try {
            this.cache.get((Object)key, () -> (String)this.cacheLoader.load((Object)key));
            return this.getFile(key);
        }
        catch (ExecutionException e) {
            LOG.error("Error loading [{}] from cache", (Object)key);
            throw new IOException(e);
        }
    }

    private void adjustSize() {
        long currentSize = this.cache.size();
        if (currentSize > this.highWaterMark) {
            this.highWaterMark = currentSize;
            while (this.highWaterMark > this.loggedWaterMark + 50000L) {
                this.loggedWaterMark += 50000L;
                LOG.info("New high water mark: {} entries", (Object)this.loggedWaterMark);
            }
        }
        if ((double)currentSize < (double)this.maxEntryCount * 0.9 && this.currentBlockLimit == this.maxBlocks) {
            return;
        }
        if (currentSize < this.maxEntryCount) {
            if ((double)currentSize >= (double)this.maxEntryCount * 0.9) {
                return;
            }
            if (this.currentBlockLimit < this.maxBlocks) {
                this.currentBlockLimit = Math.max(this.currentBlockLimit + 10L, this.cache.getUsedMemory() + 10L);
                this.currentBlockLimit = Math.min(this.currentBlockLimit, this.maxBlocks);
                LOG.debug("Grow the cache size to {}", (Object)this.currentBlockLimit);
                this.cache.setMaxMemory(this.currentBlockLimit);
            }
            return;
        }
        this.currentBlockLimit = Math.min((int)((double)this.currentBlockLimit * 0.98 - 1.0), (int)((double)this.cache.getUsedMemory() * 0.98 - 1.0));
        this.currentBlockLimit = Math.min(this.currentBlockLimit, this.maxBlocks);
        LOG.info("Shrinking the file cache size to {} because there are {} files (limit: {})", new Object[]{this.currentBlockLimit, this.cache.size(), this.maxEntryCount});
        this.cache.setMaxMemory(this.currentBlockLimit);
    }

    public void invalidate(Object key) {
        this.cache.invalidate(key);
    }

    public DataStoreCacheStatsMBean getStats() {
        return this.cacheStats;
    }

    @Override
    public void close() {
        LOG.info("Cache stats on close [{}]", (Object)this.cacheStats.cacheInfoAsString());
        new ExecutorCloser(this.executor).close();
    }

    private int build() {
        DataStoreCacheUpgradeUtils.moveDownloadCache(this.parent);
        long count = FileTreeTraverser.depthFirstPostOrder((File)this.cacheRoot).filter(file -> file.isFile() && !FilenameUtils.normalizeNoEndSeparator((String)file.getParent()).equals(this.cacheRoot.getAbsolutePath())).flatMap(toBeSyncedFile -> {
            try {
                this.put(toBeSyncedFile.getName(), (File)toBeSyncedFile, false);
                LOG.trace("Added file [{}} to in-memory cache", toBeSyncedFile);
                return Stream.of(toBeSyncedFile);
            }
            catch (Exception e) {
                LOG.error("Error in putting cached file in map[{}]", toBeSyncedFile);
                return Stream.empty();
            }
        }).count();
        LOG.trace("[{}] files put in im-memory cache", (Object)count);
        return (int)count;
    }

    private class CacheBuildJob
    implements Callable<Integer> {
        private CacheBuildJob() {
        }

        @Override
        public Integer call() {
            Stopwatch watch = Stopwatch.createStarted();
            int count = FileCache.this.build();
            LOG.info("Cache built with [{}] files from file system in [{}] seconds", (Object)count, (Object)watch.elapsed(TimeUnit.SECONDS));
            return count;
        }
    }
}

