/*
 * Decompiled with CFR 0.152.
 */
package org.apache.jackrabbit.oak.plugins.document;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.SortedMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;
import java.util.concurrent.locks.ReentrantLock;
import java.util.function.Consumer;
import org.apache.jackrabbit.guava.common.base.Predicate;
import org.apache.jackrabbit.guava.common.base.Supplier;
import org.apache.jackrabbit.guava.common.collect.Iterables;
import org.apache.jackrabbit.guava.common.collect.Lists;
import org.apache.jackrabbit.guava.common.collect.Maps;
import org.apache.jackrabbit.guava.common.collect.Sets;
import org.apache.jackrabbit.oak.commons.TimeDurationFormatter;
import org.apache.jackrabbit.oak.plugins.document.ClusterNodeInfoDocument;
import org.apache.jackrabbit.oak.plugins.document.Collection;
import org.apache.jackrabbit.oak.plugins.document.Document;
import org.apache.jackrabbit.oak.plugins.document.DocumentStore;
import org.apache.jackrabbit.oak.plugins.document.DocumentStoreException;
import org.apache.jackrabbit.oak.plugins.document.JournalEntry;
import org.apache.jackrabbit.oak.plugins.document.MissingLastRevSeeker;
import org.apache.jackrabbit.oak.plugins.document.NodeDocument;
import org.apache.jackrabbit.oak.plugins.document.NodeDocumentSweepListener;
import org.apache.jackrabbit.oak.plugins.document.NodeDocumentSweeper;
import org.apache.jackrabbit.oak.plugins.document.Path;
import org.apache.jackrabbit.oak.plugins.document.RecoveryContext;
import org.apache.jackrabbit.oak.plugins.document.Revision;
import org.apache.jackrabbit.oak.plugins.document.RevisionContext;
import org.apache.jackrabbit.oak.plugins.document.UnsavedModifications;
import org.apache.jackrabbit.oak.plugins.document.UpdateOp;
import org.apache.jackrabbit.oak.plugins.document.cache.CacheInvalidationStats;
import org.apache.jackrabbit.oak.plugins.document.util.MapFactory;
import org.apache.jackrabbit.oak.plugins.document.util.Utils;
import org.apache.jackrabbit.oak.stats.Clock;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class LastRevRecoveryAgent {
    private final Logger log = LoggerFactory.getLogger(this.getClass());
    private final DocumentStore store;
    private final RevisionContext revisionContext;
    private final MissingLastRevSeeker missingLastRevUtil;
    private final Consumer<Integer> afterRecovery;
    private static final long LOGINTERVALMS = TimeUnit.MINUTES.toMillis(1L);
    private static final int PSEUDO_BRANCH_COMMIT_UPDATE_OP_THRESHOLD_BYTES = 0x100000;
    private static final int PSEUDO_BRANCH_COMMIT_FLUSH_CHECK_COUNT = 4096;

    public LastRevRecoveryAgent(DocumentStore store, RevisionContext revisionContext, MissingLastRevSeeker seeker, Consumer<Integer> afterRecovery) {
        this.store = store;
        this.revisionContext = revisionContext;
        this.missingLastRevUtil = seeker;
        this.afterRecovery = afterRecovery;
    }

    public LastRevRecoveryAgent(DocumentStore store, RevisionContext context) {
        this(store, context, new MissingLastRevSeeker(store, context.getClock()), i -> {});
    }

    public int recover(int clusterId, long waitUntil) throws DocumentStoreException {
        ClusterNodeInfoDocument nodeInfo = this.missingLastRevUtil.getClusterNodeInfo(clusterId);
        if (nodeInfo != null) {
            ClusterNodeInfoDocument me;
            long now = this.revisionContext.getClock().getTime();
            if (clusterId != this.revisionContext.getClusterId() && (me = this.missingLastRevUtil.getClusterNodeInfo(this.revisionContext.getClusterId())) != null && me.isRecoveryNeeded(now)) {
                String msg = String.format("Own clusterId %s has a leaseEnd %s (%s) older than current time %s (%s). Refusing to run recovery on clusterId %s.", this.revisionContext.getClusterId(), me.getLeaseEndTime(), Utils.asISO8601(me.getLeaseEndTime()), now, Utils.asISO8601(now), clusterId);
                throw new DocumentStoreException(msg);
            }
            if (nodeInfo.isRecoveryNeeded(now)) {
                Revision bc;
                Object reason;
                long startTime;
                NodeDocument root = this.missingLastRevUtil.getRoot();
                Revision lastRev = root.getLastRev().get(clusterId);
                Revision sweepRev = root.getSweepRevisions().getRevision(clusterId);
                if (lastRev != null) {
                    startTime = lastRev.getTimestamp();
                    reason = "lastRev: " + lastRev.toString();
                } else {
                    startTime = nodeInfo.getStartTime();
                    reason = String.format("no lastRev for root, using startTime %d", startTime);
                }
                if (sweepRev != null && sweepRev.getTimestamp() < startTime) {
                    startTime = sweepRev.getTimestamp();
                    reason = "sweepRev: " + sweepRev.toString();
                }
                if ((bc = this.getEarliestBranchCommitMergedAround(root, startTime, clusterId)) != null) {
                    startTime = bc.getTimestamp();
                    reason = "branchRev: " + bc.toString();
                }
                return this.recoverCandidates(nodeInfo, startTime, waitUntil, (String)reason);
            }
        }
        this.log.debug("No recovery needed for clusterId {}", (Object)clusterId);
        return 0;
    }

    public int recover(int clusterId) throws DocumentStoreException {
        return this.recover(clusterId, 0L);
    }

    public int recover(Iterable<NodeDocument> suspects, int clusterId) throws DocumentStoreException {
        return this.recover(suspects, clusterId, false);
    }

    public int recover(Iterable<NodeDocument> suspects, final int clusterId, final boolean dryRun) throws DocumentStoreException {
        long startOfScan;
        ClusterNodeInfoDocument nodeInfo;
        long deadline = Long.MAX_VALUE;
        if (clusterId == this.revisionContext.getClusterId() && (nodeInfo = this.missingLastRevUtil.getClusterNodeInfo(clusterId)) != null && nodeInfo.isActive()) {
            deadline = nodeInfo.getLeaseEndTime() - 20000L;
        }
        NodeDocument rootDoc = Utils.getRootDocument(this.store);
        final AtomicReference<Revision> sweepRev = new AtomicReference<Revision>();
        if (rootDoc.getSweepRevisions().getRevision(clusterId) != null) {
            final RecoveryContext context = new RecoveryContext(rootDoc, this.revisionContext.getClock(), clusterId, this.revisionContext::getCommitValue);
            NodeDocumentSweeper sweeper = new NodeDocumentSweeper(context, true);
            this.log.info("Starting cache invalidation before sweep...");
            CacheInvalidationStats stats = this.store.invalidateCache(Iterables.transform(suspects, Document::getId));
            this.log.info("Invalidation stats: {}", (Object)stats);
            sweeper.sweep(suspects, new NodeDocumentSweepListener(){

                @Override
                public void sweepUpdate(Map<Path, UpdateOp> updates) throws DocumentStoreException {
                    if (dryRun) {
                        LastRevRecoveryAgent.this.log.info("Dry run of sweeper identified [{}] documents for cluster node [{}]: {}", new Object[]{updates.size(), clusterId, updates.values()});
                        return;
                    }
                    JournalEntry inv = Collection.JOURNAL.newDocument(LastRevRecoveryAgent.this.store);
                    inv.modified(updates.keySet());
                    Revision r = context.newRevision().asBranchRevision();
                    UpdateOp invOp = inv.asUpdateOp(r);
                    JournalEntry entry = Collection.JOURNAL.newDocument(LastRevRecoveryAgent.this.store);
                    entry.invalidate(Collections.singleton(r));
                    Revision jRev = context.newRevision();
                    UpdateOp jOp = entry.asUpdateOp(jRev);
                    if (!LastRevRecoveryAgent.this.store.create(Collection.JOURNAL, Lists.newArrayList((Object[])new UpdateOp[]{invOp, jOp}))) {
                        String msg = "Unable to create journal entries for document invalidation.";
                        throw new DocumentStoreException(msg);
                    }
                    sweepRev.set(Utils.max((Revision)sweepRev.get(), jRev));
                    LastRevRecoveryAgent.this.store.createOrUpdate(Collection.NODES, Lists.newArrayList(updates.values()));
                    LastRevRecoveryAgent.this.log.info("Sweeper updated {}", updates.keySet());
                }
            });
            if (sweepRev.get() != null) {
                sweepRev.set(Utils.max((Revision)sweepRev.get(), context.newRevision()));
            }
        }
        UnsavedModifications unsaved = new UnsavedModifications();
        UnsavedModifications unsavedParents = new UnsavedModifications();
        ConcurrentMap<Path, Revision> knownLastRevOrModification = MapFactory.getInstance().create();
        JournalEntry changes = Collection.JOURNAL.newDocument(this.store);
        Clock clock = this.revisionContext.getClock();
        long totalCount = 0L;
        long lastCount = 0L;
        long lastLog = startOfScan = clock.getTime();
        ArrayList<Revision> pseudoBcRevs = new ArrayList<Revision>();
        int nextFlushCheckCount = 4096;
        for (NodeDocument doc : suspects) {
            ++totalCount;
            ++lastCount;
            long now = clock.getTime();
            long lastElapsed = now - lastLog;
            if (lastElapsed >= LOGINTERVALMS) {
                TimeDurationFormatter df = TimeDurationFormatter.forLogging();
                long totalElapsed = now - startOfScan;
                long totalRateMin = totalCount * TimeUnit.MINUTES.toMillis(1L) / totalElapsed;
                long lastRateMin = lastCount * TimeUnit.MINUTES.toMillis(1L) / lastElapsed;
                String message = String.format("Recovery for cluster node [%d]: %d nodes scanned in %s (~%d/m) - last interval %d nodes in %s (~%d/m)", clusterId, totalCount, df.format(totalElapsed, TimeUnit.MILLISECONDS), totalRateMin, lastCount, df.format(lastElapsed, TimeUnit.MILLISECONDS), lastRateMin);
                this.log.info(message);
                lastLog = now;
                lastCount = 0L;
            }
            Revision currentLastRev = doc.getLastRev().get(clusterId);
            Revision lastModifiedRev = this.determineLastModification(doc, clusterId);
            Revision lastRevForParents = Utils.max(lastModifiedRev, currentLastRev);
            if (lastRevForParents != null) {
                knownLastRevOrModification.put(doc.getPath(), lastRevForParents);
            }
            if (lastRevForParents != null) {
                Path path = doc.getPath();
                changes.modified(path);
                while ((path = path.getParent()) != null) {
                    unsavedParents.put(path, lastRevForParents);
                }
            }
            if (changes.getNumChangedNodes() < nextFlushCheckCount) continue;
            Revision pseudoBcRev = Revision.newRevision(clusterId).asBranchRevision();
            UpdateOp pseudoBcUpdateOp = changes.asUpdateOp(pseudoBcRev);
            int approxPseudoBcUpdateOpSize = pseudoBcUpdateOp.toString().length();
            if (approxPseudoBcUpdateOpSize >= 0x100000) {
                if (this.store.create(Collection.JOURNAL, Collections.singletonList(pseudoBcUpdateOp))) {
                    this.log.info("recover : created intermediate pseudo-bc journal entry with rev {} and approx size {} bytes.", (Object)pseudoBcRev, (Object)approxPseudoBcUpdateOpSize);
                    pseudoBcRevs.add(pseudoBcRev);
                    changes = Collection.JOURNAL.newDocument(this.store);
                    nextFlushCheckCount = 4096;
                    continue;
                }
                this.log.warn("recover : could not create intermediate pseudo-bc journal entry with rev {}", (Object)pseudoBcRev);
                nextFlushCheckCount += changes.getNumChangedNodes() + 2048;
                continue;
            }
            nextFlushCheckCount = changes.getNumChangedNodes() + 4096;
        }
        changes.branchCommit(pseudoBcRevs);
        for (Path parentPath : unsavedParents.getPaths()) {
            ArrayList<Path> missingDocuments;
            NodeDocument doc;
            Revision calcLastRev = unsavedParents.get(parentPath);
            Revision knownLastRev = (Revision)knownLastRevOrModification.get(parentPath);
            if (knownLastRev == null && (doc = this.findNearestAncestorOrSelf(parentPath, missingDocuments = new ArrayList<Path>())) != null) {
                Revision lastRev = doc.getLastRev().get(clusterId);
                Revision lastMod = this.determineLastModification(doc, clusterId);
                knownLastRev = Utils.max(lastRev, lastMod);
                if (!missingDocuments.isEmpty() && doc.getLocalMap(":doc-pattern").isEmpty()) {
                    for (Path p : missingDocuments) {
                        this.log.warn("Unable to find document: {}", (Object)Utils.getIdFromPath(p));
                    }
                }
            }
            if (knownLastRev != null && calcLastRev.compareRevisionTime(knownLastRev) <= 0) continue;
            unsaved.put(parentPath, calcLastRev);
        }
        if (sweepRev.get() != null) {
            unsaved.put(Path.ROOT, (Revision)sweepRev.get());
        }
        final Revision lastRootRev = unsaved.get(Path.ROOT);
        int size = unsaved.getPaths().size();
        String updates = unsaved.toString();
        if (dryRun) {
            this.log.info("Dry run of lastRev recovery identified [{}] documents for cluster node [{}]: {}", new Object[]{size, clusterId, updates});
        } else {
            if (clock.getTime() > deadline) {
                String msg = String.format("Cluster node %d was unable to perform lastRev recovery for clusterId %d within deadline: %s", clusterId, clusterId, Utils.timestampToString(deadline));
                throw new DocumentStoreException(msg);
            }
            final JournalEntry finalChanges = changes;
            unsaved.persist(this.store, new Supplier<Revision>(){

                public Revision get() {
                    return (Revision)sweepRev.get();
                }
            }, new UnsavedModifications.Snapshot(){

                @Override
                public void acquiring(Revision mostRecent) {
                    if (lastRootRev == null) {
                        return;
                    }
                    String id = JournalEntry.asId(lastRootRev);
                    JournalEntry existingEntry = LastRevRecoveryAgent.this.store.find(Collection.JOURNAL, id);
                    if (existingEntry != null) {
                        LastRevRecoveryAgent.this.log.warn("Journal entry {} already exists", (Object)id);
                        return;
                    }
                    if (LastRevRecoveryAgent.this.store.create(Collection.JOURNAL, Collections.singletonList(finalChanges.asUpdateOp(lastRootRev)))) {
                        LastRevRecoveryAgent.this.log.info("Recovery created journal entry {}", (Object)id);
                    } else {
                        LastRevRecoveryAgent.this.log.warn("Unable to create journal entry {} (already exists).", (Object)id);
                    }
                }
            }, new ReentrantLock());
            this.log.info("Updated lastRev of [{}] documents while performing lastRev recovery for cluster node [{}]: {}", new Object[]{size, clusterId, updates});
        }
        return size;
    }

    @Nullable
    private Revision getEarliestBranchCommitMergedAround(@NotNull NodeDocument doc, long timeMillis, int clusterId) {
        Revision earliest = null;
        for (Revision bc : doc.getLocalBranchCommits()) {
            Revision mergeRevision;
            String cv;
            if (bc.getClusterId() != clusterId || !Utils.isCommitted(cv = this.revisionContext.getCommitValue(bc, doc)) || (mergeRevision = Utils.resolveCommitRevision(bc, cv)).getTimestamp() <= timeMillis || bc.getTimestamp() >= timeMillis) continue;
            earliest = Utils.min(earliest, bc);
        }
        return earliest;
    }

    @Nullable
    private NodeDocument findNearestAncestorOrSelf(@NotNull Path path, @NotNull List<Path> missingDocuments) {
        NodeDocument ancestor;
        while ((ancestor = this.store.find(Collection.NODES, Utils.getIdFromPath(path))) == null) {
            missingDocuments.add(path);
            if ((path = path.getParent()) != null) continue;
            break;
        }
        return ancestor;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private int recoverCandidates(ClusterNodeInfoDocument nodeInfo, long startTime, long waitUntil, String info) throws DocumentStoreException {
        ClusterNodeInfoDocument infoDoc = nodeInfo;
        int clusterId = infoDoc.getClusterId();
        while (!this.missingLastRevUtil.acquireRecoveryLock(clusterId, this.revisionContext.getClusterId())) {
            Clock clock = this.revisionContext.getClock();
            long remaining = waitUntil - clock.getTime();
            if (remaining < 0L) {
                return -1;
            }
            this.log.info("Last revision recovery already being performed by cluster node {}. Waiting at most until {} for recovery to finish ({} seconds remaining).", new Object[]{infoDoc.getRecoveryBy(), Utils.timestampToString(waitUntil), remaining / 1000L});
            long time = Math.min(waitUntil, clock.getTime() + 5000L);
            try {
                clock.waitUntil(time);
            }
            catch (InterruptedException e) {
                Thread.interrupted();
                String msg = "Interrupted while waiting for _lastRev recovery to finish.";
                throw new DocumentStoreException(msg, e);
            }
            infoDoc = this.missingLastRevUtil.getClusterNodeInfo(clusterId);
            if (infoDoc == null) {
                String msg = String.format("No cluster node info document for id %d", clusterId);
                throw new DocumentStoreException(msg);
            }
            if (infoDoc.isRecoveryNeeded(clock.getTime())) continue;
            return 0;
        }
        boolean success = false;
        try {
            int n;
            this.log.info("Recovering candidates modified after: [{}] for clusterId [{}] [{}]", new Object[]{Utils.timestampToString(startTime), clusterId, info});
            Iterable<NodeDocument> suspects = this.missingLastRevUtil.getCandidates(startTime);
            try {
                this.log.info("Performing Last Revision Recovery for clusterNodeId {}", (Object)clusterId);
                int num = this.recover(suspects, clusterId);
                success = true;
                n = num;
            }
            catch (Throwable throwable) {
                Utils.closeIfCloseable(suspects);
                throw throwable;
            }
            Utils.closeIfCloseable(suspects);
            return n;
        }
        finally {
            this.missingLastRevUtil.releaseRecoveryLock(clusterId, success);
            this.afterRecovery.accept(clusterId);
        }
    }

    @Nullable
    private Revision determineLastModification(NodeDocument doc, int clusterId) {
        ClusterPredicate cp = new ClusterPredicate(clusterId);
        Revision lastModified = null;
        block0: for (String property : Sets.filter(doc.keySet(), Utils.PROPERTY_OR_DELETED)) {
            SortedMap<Revision, String> valueMap = doc.getLocalMap(property);
            for (Map.Entry entry : Maps.filterKeys(valueMap, (Predicate)cp).entrySet()) {
                Revision rev = (Revision)entry.getKey();
                String cv = this.revisionContext.getCommitValue(rev, doc);
                if (!Utils.isCommitted(cv)) continue;
                lastModified = Utils.max(lastModified, Utils.resolveCommitRevision(rev, cv));
                continue block0;
            }
        }
        return lastModified;
    }

    public boolean isRecoveryNeeded() {
        return this.missingLastRevUtil.isRecoveryNeeded();
    }

    public void performRecoveryIfNeeded() {
        if (this.isRecoveryNeeded()) {
            Iterable<Integer> clusterIds = this.getRecoveryCandidateNodes();
            this.log.info("ClusterNodeId [{}] starting Last Revision Recovery for clusterNodeId(s) {}", (Object)this.revisionContext.getClusterId(), clusterIds);
            for (int clusterId : clusterIds) {
                if (this.recover(clusterId) != -1) continue;
                this.log.info("Last Revision Recovery for cluster node {} ongoing by other cluster node.", (Object)clusterId);
            }
        }
    }

    public Iterable<Integer> getRecoveryCandidateNodes() {
        return Iterables.transform((Iterable)Iterables.filter(this.missingLastRevUtil.getAllClusters(), (Predicate)new Predicate<ClusterNodeInfoDocument>(){

            public boolean apply(ClusterNodeInfoDocument input) {
                return LastRevRecoveryAgent.this.revisionContext.getClusterId() != input.getClusterId() && input.isRecoveryNeeded(LastRevRecoveryAgent.this.revisionContext.getClock().getTime());
            }
        }), ClusterNodeInfoDocument::getClusterId);
    }

    private static class ClusterPredicate
    implements Predicate<Revision> {
        private final int clusterId;

        private ClusterPredicate(int clusterId) {
            this.clusterId = clusterId;
        }

        public boolean apply(Revision input) {
            return this.clusterId == input.getClusterId();
        }
    }
}

