/*
 * Decompiled with CFR 0.152.
 */
package org.apache.jackrabbit.oak.segment.file.tar;

import java.io.EOFException;
import java.io.File;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.channels.FileChannel;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.TimeUnit;
import java.util.zip.CRC32;
import org.apache.jackrabbit.guava.common.base.Charsets;
import org.apache.jackrabbit.guava.common.base.Preconditions;
import org.apache.jackrabbit.guava.common.base.Stopwatch;
import org.apache.jackrabbit.oak.commons.Buffer;
import org.apache.jackrabbit.oak.segment.file.tar.index.IndexEntry;
import org.apache.jackrabbit.oak.segment.file.tar.index.IndexWriter;
import org.apache.jackrabbit.oak.segment.file.tar.index.SimpleIndexEntry;
import org.apache.jackrabbit.oak.segment.spi.monitor.FileStoreMonitor;
import org.apache.jackrabbit.oak.segment.spi.monitor.IOMonitor;
import org.apache.jackrabbit.oak.segment.spi.persistence.SegmentArchiveWriter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SegmentTarWriter
implements SegmentArchiveWriter {
    private static final Logger log = LoggerFactory.getLogger(SegmentTarWriter.class);
    private static final byte[] ZERO_BYTES = new byte[512];
    private final FileStoreMonitor monitor;
    private final File file;
    private final IOMonitor ioMonitor;
    private final Map<UUID, IndexEntry> index = Collections.synchronizedMap(new LinkedHashMap());
    private RandomAccessFile access = null;
    private FileChannel channel = null;
    private volatile long length;

    public SegmentTarWriter(File file, FileStoreMonitor monitor, IOMonitor ioMonitor) {
        this.file = file;
        this.monitor = monitor;
        this.ioMonitor = ioMonitor;
    }

    @Override
    public void writeSegment(long msb, long lsb, byte[] data, int offset, int size, int generation, int fullGeneration, boolean compacted) throws IOException {
        UUID uuid = new UUID(msb, lsb);
        CRC32 checksum = new CRC32();
        checksum.update(data, offset, size);
        String entryName = String.format("%s.%08x", uuid, checksum.getValue());
        byte[] header = SegmentTarWriter.newEntryHeader(entryName, size);
        log.debug("Writing segment {} to {}", (Object)uuid, (Object)this.file);
        if (this.access == null) {
            this.access = new RandomAccessFile(this.file, "rw");
            this.channel = this.access.getChannel();
        }
        int padding = SegmentTarWriter.getPaddingSize(size);
        long initialLength = this.access.getFilePointer();
        this.access.write(header);
        long dataOffset = this.access.getFilePointer();
        this.ioMonitor.beforeSegmentWrite(this.file, msb, lsb, size);
        Stopwatch stopwatch = Stopwatch.createStarted();
        this.access.write(data, offset, size);
        this.ioMonitor.afterSegmentWrite(this.file, msb, lsb, size, stopwatch.elapsed(TimeUnit.NANOSECONDS));
        if (padding > 0) {
            this.access.write(ZERO_BYTES, 0, padding);
        }
        long currentLength = this.access.getFilePointer();
        this.monitor.written(currentLength - initialLength);
        this.length = currentLength;
        this.index.put(new UUID(msb, lsb), new SimpleIndexEntry(msb, lsb, (int)dataOffset, size, generation, fullGeneration, compacted));
    }

    @Override
    public Buffer readSegment(long msb, long lsb) throws IOException {
        IndexEntry indexEntry = this.index.get(new UUID(msb, lsb));
        if (indexEntry == null) {
            return null;
        }
        Preconditions.checkState((this.channel != null ? 1 : 0) != 0);
        Buffer data = Buffer.allocate((int)indexEntry.getLength());
        if (data.readFully(this.channel, indexEntry.getPosition()) < indexEntry.getLength()) {
            throw new EOFException();
        }
        data.rewind();
        return data;
    }

    @Override
    public boolean containsSegment(long msb, long lsb) {
        return this.index.containsKey(new UUID(msb, lsb));
    }

    @Override
    public void writeGraph(byte[] data) throws IOException {
        int paddingSize = SegmentTarWriter.getPaddingSize(data.length);
        byte[] header = SegmentTarWriter.newEntryHeader(this.file.getName() + ".gph", data.length + paddingSize);
        this.access.write(header);
        if (paddingSize > 0) {
            this.access.write(ZERO_BYTES, 0, paddingSize);
        }
        this.access.write(data);
        this.monitor.written(header.length + paddingSize + data.length);
        this.length = this.access.getFilePointer();
    }

    @Override
    public void writeBinaryReferences(byte[] data) throws IOException {
        int paddingSize = SegmentTarWriter.getPaddingSize(data.length);
        byte[] header = SegmentTarWriter.newEntryHeader(this.file.getName() + ".brf", data.length + paddingSize);
        this.access.write(header);
        if (paddingSize > 0) {
            this.access.write(ZERO_BYTES, 0, paddingSize);
        }
        this.access.write(data);
        this.monitor.written(header.length + paddingSize + data.length);
        this.length = this.access.getFilePointer();
    }

    @Override
    public long getLength() {
        return this.length;
    }

    @Override
    public int getEntryCount() {
        return this.index.size();
    }

    private void writeIndex() throws IOException {
        IndexWriter writer = IndexWriter.newIndexWriter(512);
        for (IndexEntry entry : this.index.values()) {
            writer.addEntry(entry.getMsb(), entry.getLsb(), entry.getPosition(), entry.getLength(), entry.getGeneration(), entry.getFullGeneration(), entry.isCompacted());
        }
        byte[] data = writer.write();
        byte[] header = SegmentTarWriter.newEntryHeader(this.file.getName() + ".idx", data.length);
        this.access.write(header);
        this.access.write(data);
        this.monitor.written(header.length + data.length);
        this.length = this.access.getFilePointer();
    }

    @Override
    public void close() throws IOException {
        this.writeIndex();
        this.access.write(ZERO_BYTES);
        this.access.write(ZERO_BYTES);
        this.access.close();
        this.monitor.written(1024L);
    }

    @Override
    public boolean isCreated() {
        return this.access != null;
    }

    @Override
    public void flush() throws IOException {
        this.access.getFD().sync();
    }

    @Override
    public String getName() {
        return this.file.getName();
    }

    @Override
    public boolean isRemote() {
        return false;
    }

    private static byte[] newEntryHeader(String name, int size) {
        byte[] header = new byte[512];
        byte[] nameBytes = name.getBytes(Charsets.UTF_8);
        System.arraycopy(nameBytes, 0, header, 0, Math.min(nameBytes.length, 100));
        System.arraycopy(String.format("%07o", 256).getBytes(Charsets.UTF_8), 0, header, 100, 7);
        System.arraycopy(String.format("%07o", 0).getBytes(Charsets.UTF_8), 0, header, 108, 7);
        System.arraycopy(String.format("%07o", 0).getBytes(Charsets.UTF_8), 0, header, 116, 7);
        System.arraycopy(String.format("%011o", size).getBytes(Charsets.UTF_8), 0, header, 124, 11);
        long time = System.currentTimeMillis() / 1000L;
        System.arraycopy(String.format("%011o", time).getBytes(Charsets.UTF_8), 0, header, 136, 11);
        System.arraycopy(new byte[]{32, 32, 32, 32, 32, 32, 32, 32}, 0, header, 148, 8);
        header[156] = 48;
        int checksum = 0;
        for (byte aHeader : header) {
            checksum += aHeader & 0xFF;
        }
        System.arraycopy(String.format("%06o\u0000 ", checksum).getBytes(Charsets.UTF_8), 0, header, 148, 8);
        return header;
    }

    static int getPaddingSize(int size) {
        int remainder = size % 512;
        if (remainder > 0) {
            return 512 - remainder;
        }
        return 0;
    }
}

