/*
 * Decompiled with CFR 0.152.
 */
package io.smallrye.faulttolerance.core.circuit.breaker;

import io.smallrye.faulttolerance.core.FaultToleranceStrategy;
import io.smallrye.faulttolerance.core.InvocationContext;
import io.smallrye.faulttolerance.core.circuit.breaker.CircuitBreakerEvents;
import io.smallrye.faulttolerance.core.circuit.breaker.CircuitBreakerLogger;
import io.smallrye.faulttolerance.core.circuit.breaker.RollingWindow;
import io.smallrye.faulttolerance.core.stopwatch.RunningStopwatch;
import io.smallrye.faulttolerance.core.stopwatch.Stopwatch;
import io.smallrye.faulttolerance.core.util.Preconditions;
import io.smallrye.faulttolerance.core.util.SetOfThrowables;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;
import org.eclipse.microprofile.faulttolerance.exceptions.CircuitBreakerOpenException;

public class CircuitBreaker<V>
implements FaultToleranceStrategy<V> {
    public static final int STATE_CLOSED = 0;
    public static final int STATE_OPEN = 1;
    public static final int STATE_HALF_OPEN = 2;
    final FaultToleranceStrategy<V> delegate;
    final String description;
    final SetOfThrowables failOn;
    final SetOfThrowables skipOn;
    final long delayInMillis;
    final int rollingWindowSize;
    final int failureThreshold;
    final int successThreshold;
    final Stopwatch stopwatch;
    final AtomicReference<State> state;

    public CircuitBreaker(FaultToleranceStrategy<V> delegate, String description, SetOfThrowables failOn, SetOfThrowables skipOn, long delayInMillis, int requestVolumeThreshold, double failureRatio, int successThreshold, Stopwatch stopwatch) {
        this.delegate = Preconditions.checkNotNull(delegate, "Circuit breaker delegate must be set");
        this.description = Preconditions.checkNotNull(description, "Circuit breaker description must be set");
        this.failOn = Preconditions.checkNotNull(failOn, "Set of fail-on throwables must be set");
        this.skipOn = Preconditions.checkNotNull(skipOn, "Set of skip-on throwables must be set");
        this.delayInMillis = Preconditions.check(delayInMillis, delayInMillis >= 0L, "Circuit breaker delay must be >= 0");
        this.successThreshold = Preconditions.check(successThreshold, successThreshold > 0, "Circuit breaker success threshold must be > 0");
        this.stopwatch = Preconditions.checkNotNull(stopwatch, "Stopwatch must be set");
        this.failureThreshold = Preconditions.check((int)Math.ceil(failureRatio * (double)requestVolumeThreshold), failureRatio >= 0.0 && failureRatio <= 1.0, "Circuit breaker rolling window failure ratio must be >= 0 && <= 1");
        this.rollingWindowSize = Preconditions.check(requestVolumeThreshold, requestVolumeThreshold > 0, "Circuit breaker rolling window size must be > 0");
        this.state = new AtomicReference<State>(State.closed(this.rollingWindowSize, this.failureThreshold));
    }

    @Override
    public V apply(InvocationContext<V> ctx) throws Exception {
        CircuitBreakerLogger.LOG.trace("CircuitBreaker started");
        try {
            V v = this.doApply(ctx);
            return v;
        }
        finally {
            CircuitBreakerLogger.LOG.trace("CircuitBreaker finished");
        }
    }

    private V doApply(InvocationContext<V> ctx) throws Exception {
        State state = this.state.get();
        switch (state.id) {
            case 0: {
                return this.inClosed(ctx, state);
            }
            case 1: {
                return this.inOpen(ctx, state);
            }
            case 2: {
                return this.inHalfOpen(ctx, state);
            }
        }
        throw new AssertionError((Object)("Invalid circuit breaker state: " + state.id));
    }

    boolean isConsideredSuccess(Throwable e) {
        return this.skipOn.includes(e.getClass()) || !this.failOn.includes(e.getClass());
    }

    private V inClosed(InvocationContext<V> ctx, State state) throws Exception {
        try {
            CircuitBreakerLogger.LOG.trace("Circuit breaker closed, invocation allowed");
            V result = this.delegate.apply(ctx);
            this.inClosedHandleResult(true, ctx, state);
            return result;
        }
        catch (Throwable e) {
            this.inClosedHandleResult(this.isConsideredSuccess(e), ctx, state);
            throw e;
        }
    }

    final void inClosedHandleResult(boolean isSuccess, InvocationContext<V> ctx, State state) {
        boolean failureThresholdReached;
        ctx.fireEvent(isSuccess ? CircuitBreakerEvents.Finished.SUCCESS : CircuitBreakerEvents.Finished.FAILURE);
        boolean bl = failureThresholdReached = isSuccess ? state.rollingWindow.recordSuccess() : state.rollingWindow.recordFailure();
        if (failureThresholdReached) {
            CircuitBreakerLogger.LOG.trace("Failure threshold reached, circuit breaker moving to open");
            this.toOpen(ctx, state);
        }
    }

    private V inOpen(InvocationContext<V> ctx, State state) throws Exception {
        if (state.runningStopwatch.elapsedTimeInMillis() < this.delayInMillis) {
            CircuitBreakerLogger.LOG.trace("Circuit breaker open, invocation prevented");
            ctx.fireEvent(CircuitBreakerEvents.Finished.PREVENTED);
            throw new CircuitBreakerOpenException(this.description + " circuit breaker is open");
        }
        CircuitBreakerLogger.LOG.trace("Delay elapsed, circuit breaker moving to half-open");
        this.toHalfOpen(ctx, state);
        return this.doApply(ctx);
    }

    private V inHalfOpen(InvocationContext<V> ctx, State state) throws Exception {
        if (state.probeAttempts.incrementAndGet() > this.successThreshold) {
            CircuitBreakerLogger.LOG.trace("Circuit breaker half-open, invocation prevented");
            ctx.fireEvent(CircuitBreakerEvents.Finished.PREVENTED);
            throw new CircuitBreakerOpenException(this.description + " circuit breaker is half-open");
        }
        try {
            CircuitBreakerLogger.LOG.trace("Circuit breaker half-open, probe invocation allowed");
            V result = this.delegate.apply(ctx);
            this.inHalfOpenHandleResult(true, ctx, state);
            return result;
        }
        catch (Throwable e) {
            this.inHalfOpenHandleResult(this.isConsideredSuccess(e), ctx, state);
            throw e;
        }
    }

    final void inHalfOpenHandleResult(boolean isSuccess, InvocationContext<V> ctx, State state) {
        ctx.fireEvent(isSuccess ? CircuitBreakerEvents.Finished.SUCCESS : CircuitBreakerEvents.Finished.FAILURE);
        if (isSuccess) {
            int successes = state.consecutiveSuccesses.incrementAndGet();
            if (successes >= this.successThreshold) {
                CircuitBreakerLogger.LOG.trace("Success threshold reached, circuit breaker moving to closed");
                this.toClosed(ctx, state);
            }
        } else {
            CircuitBreakerLogger.LOG.trace("Failure while in half-open, circuit breaker moving to open");
            this.toOpen(ctx, state);
        }
    }

    void toClosed(InvocationContext<V> ctx, State state) {
        State newState = State.closed(this.rollingWindowSize, this.failureThreshold);
        boolean moved = this.state.compareAndSet(state, newState);
        if (moved) {
            ctx.fireEvent(CircuitBreakerEvents.StateTransition.TO_CLOSED);
        }
    }

    void toOpen(InvocationContext<V> ctx, State state) {
        State newState = State.open(this.stopwatch);
        boolean moved = this.state.compareAndSet(state, newState);
        if (moved) {
            ctx.fireEvent(CircuitBreakerEvents.StateTransition.TO_OPEN);
        }
    }

    void toHalfOpen(InvocationContext<V> ctx, State state) {
        State newState = State.halfOpen();
        boolean moved = this.state.compareAndSet(state, newState);
        if (moved) {
            ctx.fireEvent(CircuitBreakerEvents.StateTransition.TO_HALF_OPEN);
        }
    }

    public int currentState() {
        return this.state.get().id;
    }

    public void reset() {
        State newState = State.closed(this.rollingWindowSize, this.failureThreshold);
        this.state.set(newState);
    }

    static final class State {
        final int id;
        RollingWindow rollingWindow;
        RunningStopwatch runningStopwatch;
        AtomicInteger probeAttempts;
        AtomicInteger consecutiveSuccesses;

        private State(int id) {
            this.id = id;
        }

        static State closed(int rollingWindowSize, int failureThreshold) {
            State result = new State(0);
            result.rollingWindow = RollingWindow.create(rollingWindowSize, failureThreshold);
            return result;
        }

        static State open(Stopwatch stopwatch) {
            State result = new State(1);
            result.runningStopwatch = stopwatch.start();
            return result;
        }

        static State halfOpen() {
            State result = new State(2);
            result.probeAttempts = new AtomicInteger(0);
            result.consecutiveSuccesses = new AtomicInteger(0);
            return result;
        }
    }
}

